<?php
require_once 'config.php';
include 'security.php';
require_once 'includes/openai_helper.php';

// Get the logged-in user's ID
$user_id = isset($_SESSION['user_id']) ? intval($_SESSION['user_id']) : 0;

if($user_id <= 0) {
    header('Location: login.php');
    exit();
}

$message = '';
$message_type = '';
$force_manual_mode = false;

// Prefill values for the form (used after AI extraction)
$prefill_income_source = '';
$prefill_amount = '';
$prefill_category = '';
$prefill_account = '';
$prefill_date = date('Y-m-d');
$prefill_notes = '';

// Handle form submission
if($_SERVER['REQUEST_METHOD'] == 'POST') {
    $mode = isset($_POST['mode']) ? $_POST['mode'] : 'manual';

    // --- Upload mode: extract data with OpenAI, then show the form prefilled ---
    if($mode === 'upload') {
        if(empty($_FILES['income-upload']['tmp_name'])) {
            $message = 'Please choose a file to upload.';
            $message_type = 'error';
        } else {
            $file = $_FILES['income-upload'];
            if($file['error'] !== UPLOAD_ERR_OK) {
                $message = 'Upload error. Please try again.';
                $message_type = 'error';
            } elseif($file['size'] > 10 * 1024 * 1024) {
                $message = 'File too large. Max 10MB.';
                $message_type = 'error';
            } else {
                $result = extract_transaction_from_file($file['tmp_name'], 'income');
                if(!empty($result['error'])) {
                    $message = $result['error'];
                    $message_type = 'error';
                    $force_manual_mode = true;
                } else {
                    $message = 'Data extracted. Please review and submit.';
                    $message_type = 'success';
                    $force_manual_mode = true;
                    $prefill_income_source = $result['description'] ?? '';
                    $prefill_amount = $result['amount'] ?? '';
                    $prefill_category = $result['category'] ?? '';
                    $prefill_account = '';
                    $prefill_date = !empty($result['date']) ? $result['date'] : date('Y-m-d');
                    $prefill_notes = '';
                }
            }
        }
    }

    // --- Manual mode: existing save flow ---
    if($mode !== 'upload') {
        $income_source = isset($_POST['income-source']) ? trim($_POST['income-source']) : '';
        $amount = isset($_POST['amount']) ? trim($_POST['amount']) : '';
        $category_input = isset($_POST['category-input']) ? trim($_POST['category-input']) : '';
        $category_id = 0;
        $account_id = isset($_POST['account']) ? intval($_POST['account']) : 0;
        $date_received = isset($_POST['date-received']) ? trim($_POST['date-received']) : '';
        $notes = isset($_POST['notes']) ? trim($_POST['notes']) : '';

        // Keep values in case of validation errors
        $prefill_income_source = $income_source;
        $prefill_amount = $amount;
        $prefill_category = $category_input;
        $prefill_account = $account_id;
        $prefill_date = $date_received ?: $prefill_date;
        $prefill_notes = $notes;

    // Handle category - check if input matches existing category or is new
    if(!empty($category_input)) {
        // Check if category exists (case-insensitive)
        $check_cat = $mysql->query("SELECT category_id FROM categories WHERE LOWER(label) = LOWER('" . $mysql->real_escape_string($category_input) . "') AND type = 'income' AND (user_id = " . $user_id . " OR user_id IS NULL) LIMIT 1");
        if($check_cat && $check_cat->num_rows > 0) {
            $cat_row = $check_cat->fetch_assoc();
            $category_id = intval($cat_row['category_id']);
        } else {
            // Create new category
            $create_cat_sql = "INSERT INTO categories (user_id, label, type) VALUES (" . $user_id . ", '" . $mysql->real_escape_string($category_input) . "', 'income')";
            if($mysql->query($create_cat_sql)) {
                $category_id = intval($mysql->insert_id);
            } else {
                $message = 'Error creating category: ' . $mysql->error;
                $message_type = 'error';
            }
        }
    }

    // Validate required fields
    if(empty($income_source) || empty($amount) || $category_id <= 0 || $account_id <= 0 || empty($date_received)) {
        $message = 'Please fill in all required fields.';
        $message_type = 'error';
    } else {
        // Clean amount (remove $ and commas)
        $amount_clean = preg_replace('/[^0-9.]/', '', $amount);
        $amount_clean = floatval($amount_clean);

        if($amount_clean <= 0) {
            $message = 'Please enter a valid amount.';
            $message_type = 'error';
        } else {
            // Parse date (assuming format MM/DD/YYYY)
            if(strpos($date_received, '/') !== false) {
                $date_parts = explode('/', $date_received);
                if(count($date_parts) == 3) {
                    $transaction_date = $date_parts[2] . '-' . $date_parts[0] . '-' . $date_parts[1];
                } else {
                    $transaction_date = date('Y-m-d');
                }
            } else {
                $transaction_date = $date_received;
            }

            // Verify account belongs to user
            $account_check = $mysql->query("SELECT account_id FROM accounts WHERE account_id = " . $account_id . " AND user_id = " . $user_id);
            if($account_check && $account_check->num_rows > 0) {
                // Insert transaction
                // Note: Category is stored in category_id field, NOT in the note field
                // The note field only contains the income source and optional notes
                $note_text = $income_source . ($notes ? ' - ' . $notes : '');
                $sql = "INSERT INTO transactions (account_id, category_id, amount, note, transaction_date) 
                        VALUES (" . $account_id . ", " . $category_id . ", " . $amount_clean . ", '" . $mysql->real_escape_string($note_text) . "', '" . $transaction_date . "')";

                if($mysql->query($sql)) {
                    $message = 'Income added successfully!';
                    $message_type = 'success';
                    // Clear form by redirecting
                    header('Location: add_income.php?success=1');
                    exit();
                } else {
                    $message = 'Error adding income: ' . $mysql->error;
                    $message_type = 'error';
                }
            } else {
                $message = 'Invalid account selected.';
                $message_type = 'error';
            }
        }
    }
    }
}

// Check for success message
if(isset($_GET['success']) && $_GET['success'] == 1) {
    $message = 'Income added successfully!';
    $message_type = 'success';
}

// Get income categories for dropdown
$income_categories_sql = "SELECT category_id, label FROM categories WHERE type = 'income' AND (user_id = " . $user_id . " OR user_id IS NULL) ORDER BY label";
$income_categories_result = $mysql->query($income_categories_sql);
$income_categories = [];
if($income_categories_result) {
    while($row = $income_categories_result->fetch_assoc()) {
        $income_categories[] = $row;
    }
}

// Get user's accounts
$accounts_sql = "SELECT account_id, name FROM accounts WHERE user_id = " . $user_id . " ORDER BY name";
$accounts_result = $mysql->query($accounts_sql);
$accounts = [];
if($accounts_result) {
    while($row = $accounts_result->fetch_assoc()) {
        $accounts[] = $row;
    }
}

// Get recent income transactions
$recent_income_sql = "SELECT t.note, t.amount, t.transaction_date, c.label as category_name 
                      FROM transactions t
                      LEFT JOIN categories c ON t.category_id = c.category_id
                      WHERE c.type = 'income' 
                        AND t.account_id IN (
                            SELECT account_id FROM accounts WHERE user_id = " . $user_id . "
                            UNION
                            SELECT account_id FROM shared_users WHERE user_id = " . $user_id . "
                        )
                      ORDER BY t.transaction_date DESC, t.transaction_id DESC
                      LIMIT 4";
$recent_income_result = $mysql->query($recent_income_sql);
$recent_income = [];
if($recent_income_result) {
    while($row = $recent_income_result->fetch_assoc()) {
        $recent_income[] = $row;
    }
}

// Get income summary for current month
$current_month_start = date('Y-m-01');
$income_summary_sql = "SELECT 
                        COUNT(*) as source_count,
                        SUM(t.amount) as total_income,
                        AVG(t.amount) as avg_income
                       FROM transactions t
                       LEFT JOIN categories c ON t.category_id = c.category_id
                       WHERE c.type = 'income' 
                         AND t.transaction_date >= '" . $current_month_start . "'
                         AND t.account_id IN (
                             SELECT account_id FROM accounts WHERE user_id = " . $user_id . "
                             UNION
                             SELECT account_id FROM shared_users WHERE user_id = " . $user_id . "
                         )";
$income_summary_result = $mysql->query($income_summary_sql);
$income_summary = ['source_count' => 0, 'total_income' => 0, 'avg_income' => 0];
if($income_summary_result && $income_summary_result->num_rows > 0) {
    $income_summary = $income_summary_result->fetch_assoc();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <!-- Google tag (gtag.js) -->
    <script async src="https://www.googletagmanager.com/gtag/js?id=G-P8J7Y3RS27"></script>
    <script>
      window.dataLayer = window.dataLayer || [];
      function gtag(){dataLayer.push(arguments);}
      gtag('js', new Date());
      gtag('config', 'G-P8J7Y3RS27');
    </script>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add Income - Budget Buddy</title>
    <link rel="stylesheet" href="css/style.css?v=<?php echo time(); ?>">
</head>
<body class="income-page">
<?php
$active_page = 'add_income';
include 'sidebar.php';
?>

<main class="main-content">
    <!-- Page Header -->
    <div class="page-header">
        <h1 class="page-title">Add Income</h1>
        <p class="page-subtitle">Record income and earnings</p>
    </div>

    <?php if($message): ?>
        <div class="message <?php echo $message_type; ?>" style="padding: 12px; margin-bottom: 20px; border-radius: 6px; <?php echo $message_type == 'success' ? 'background-color: #d4edda; color: #155724; border: 1px solid #c3e6cb;' : 'background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb;'; ?>">
            <?php echo htmlspecialchars($message); ?>
        </div>
    <?php endif; ?>

    <div class="content-layout">
        <!-- Left Column: Add Income Form -->
        <div class="form-section">
            <!-- Toggle Button -->
            <div style="display: flex; justify-content: flex-end; margin-bottom: 20px; gap: 10px; align-items: center;">
                <span style="font-size: 14px; color: #666;">Manual Input</span>
                <label style="position: relative; display: inline-block; width: 50px; height: 26px;">
                    <input type="checkbox" id="income-input-mode-toggle" style="opacity: 0; width: 0; height: 0;">
                    <span style="position: absolute; cursor: pointer; top: 0; left: 0; right: 0; bottom: 0; background-color: #ccc; transition: .4s; border-radius: 26px;"></span>
                    <span style="position: absolute; content: ''; height: 20px; width: 20px; left: 3px; bottom: 3px; background-color: white; transition: .4s; border-radius: 50%;"></span>
                </label>
                <span style="font-size: 14px; color: #666;">📷 Image Upload</span>
            </div>

            <!-- Manual Input Form -->
            <div id="income-manual-input-mode">
                <form id="manual-income-form" method="POST" action="add_income.php" class="income-form">
                    <input type="hidden" name="mode" value="manual">
                    <div class="form-field">
                        <label for="income-source">Income Source</label>
                        <input type="text" id="income-source" name="income-source" placeholder="e.g., Monthly Salary, Freelance Project" value="<?php echo htmlspecialchars($prefill_income_source); ?>" required>
                    </div>

                    <div class="form-field">
                        <label for="amount">Amount</label>
                        <input type="text" id="amount" name="amount" class="highlighted" placeholder="$ 0.00" value="<?php echo htmlspecialchars($prefill_amount); ?>" required>
                    </div>

                    <div class="form-field">
                        <label for="category-input">Category</label>
                        <input type="text" id="category-input" name="category-input" list="category-list" placeholder="Select or type a category name" autocomplete="off" value="<?php echo htmlspecialchars($prefill_category); ?>" required>
                        <datalist id="category-list">
                            <?php foreach($income_categories as $cat): ?>
                            <option value="<?php echo htmlspecialchars($cat['label']); ?>">
                                <?php endforeach; ?>
                        </datalist>
                    </div>

                    <div class="form-field">
                        <label for="account">Account</label>
                        <select id="account" name="account" required>
                            <option value="">Choose your bank account</option>
                            <?php foreach($accounts as $acc): ?>
                                <option value="<?php echo $acc['account_id']; ?>" <?php echo ($prefill_account == $acc['account_id']) ? 'selected' : ''; ?>><?php echo htmlspecialchars($acc['name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="form-field">
                        <label for="date-received">Date Received</label>
                        <div class="date-input-wrapper">
                            <input type="date" id="date-received" name="date-received" value="<?php echo htmlspecialchars($prefill_date ?: date('Y-m-d')); ?>" required>
                            <span class="date-icon">📅</span>
                        </div>
                    </div>

                    <div class="form-field">
                        <label for="notes">Notes (Optional)</label>
                        <textarea id="notes" name="notes" placeholder="Add any additional details..."><?php echo htmlspecialchars($prefill_notes); ?></textarea>
                    </div>

                    <button type="submit" class="submit-button">
                        <span>✓</span>
                        <span>Add Income</span>
                    </button>
                </form>
            </div>

            <!-- Image Upload Mode -->
            <div id="income-image-upload-mode" style="display: none;">
                <div class="upload-section" style="padding: 40px; max-width: 600px; margin: 0 auto;">
                    <div class="upload-header" style="margin-bottom: 24px; text-align: center;">
                        <h2 class="card-title-section" style="font-size: 20px; margin-bottom: 8px;">Upload Income Document</h2>
                        <p class="card-subtitle-section" style="font-size: 14px; color: #666;">Upload a photo of your pay stub, invoice, or receipt to automatically extract income details</p>
                    </div>

                    <form method="POST" action="add_income.php" enctype="multipart/form-data" style="display:flex; flex-direction:column; gap:16px;">
                        <input type="hidden" name="mode" value="upload">
                        <div class="upload-area" style="padding: 40px 20px; margin-bottom: 8px; border: 2px dashed #ccc; text-align: center; border-radius: 10px; cursor: pointer;">
                            <span class="upload-icon" style="font-size: 48px; margin-bottom: 12px; display:block;">📄</span>
                            <div class="upload-text" style="font-size: 16px; margin-bottom: 4px;">Drag & drop or browse</div>
                            <div class="upload-hint" style="font-size: 12px; color: #999;">PNG, JPG, PDF (max 10MB)</div>
                            <input type="file" name="income-upload" accept=".png,.jpg,.jpeg,.pdf" style="margin-top:12px;" required>
                        </div>

                        <button type="submit" class="submit-button" style="justify-content:center;">
                            <span>✨</span>
                            <span>Extract & Prefill</span>
                        </button>
                    </form>
                </div>
            </div>
        </div>

        <!-- Right Column: Recent Income & Summary -->
        <div class="sidebar-section">
            <!-- Recent Income Card -->
            <div class="recent-income-card">
                <div class="card-header-section">
                    <div class="card-title-section">Recent Income</div>
                    <div class="card-subtitle-section">Your latest income entries</div>
                </div>

                <div class="income-list">
                    <?php if(count($recent_income) > 0): ?>
                        <?php foreach($recent_income as $income): ?>
                            <div class="income-item">
                                <div class="income-item-info">
                                    <div class="income-item-title"><?php echo htmlspecialchars($income['note']); ?></div>
                                    <div class="income-item-meta"><?php echo date('Y-m-d', strtotime($income['transaction_date'])); ?> • <?php echo htmlspecialchars($income['category_name']); ?></div>
                                </div>
                                <div class="income-item-amount">+$<?php echo number_format($income['amount'], 2); ?></div>
                            </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <div style="padding: 20px; text-align: center; color: #666;">
                            <p>No recent income entries</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Income Summary Card -->
            <div class="summary-card">
                <div class="card-header-section">
                    <div class="card-title-section">Income Summary</div>
                    <div class="card-subtitle-section">This month</div>
                </div>

                <div class="summary-stats">
                    <div class="summary-stat">
                        <span class="summary-stat-label">Total Income</span>
                        <span class="summary-stat-value income-value">+$<?php echo number_format($income_summary['total_income'] ?? 0, 2); ?></span>
                    </div>

                    <div class="summary-stat">
                        <span class="summary-stat-label">Income Sources</span>
                        <span class="summary-stat-value"><?php echo $income_summary['source_count'] ?? 0; ?></span>
                    </div>

                    <div class="summary-stat">
                        <span class="summary-stat-label">Average per Source</span>
                        <span class="summary-stat-value">$<?php echo number_format($income_summary['avg_income'] ?? 0, 2); ?></span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</main>
<script>
    // Toggle between manual input and image upload modes
    const incomeToggle = document.getElementById('income-input-mode-toggle');
    const incomeManualMode = document.getElementById('income-manual-input-mode');
    const incomeImageMode = document.getElementById('income-image-upload-mode');
    const incomeToggleSlider = incomeToggle.nextElementSibling;
    const incomeToggleCircle = incomeToggleSlider.nextElementSibling;

    incomeToggle.addEventListener('change', function() {
        if(this.checked) {
            // Switch to image upload mode
            incomeManualMode.style.display = 'none';
            incomeImageMode.style.display = 'block';
            incomeToggleSlider.style.backgroundColor = '#4CAF50';
            incomeToggleCircle.style.transform = 'translateX(24px)';
        } else {
            // Switch to manual input mode
            incomeManualMode.style.display = 'block';
            incomeImageMode.style.display = 'none';
            incomeToggleSlider.style.backgroundColor = '#ccc';
            incomeToggleCircle.style.transform = 'translateX(0)';
        }
    });

    // If we just extracted data, force manual mode to review
    const forceManual = <?php echo $force_manual_mode ? 'true' : 'false'; ?>;
    if(forceManual) {
        incomeToggle.checked = false;
        incomeManualMode.style.display = 'block';
        incomeImageMode.style.display = 'none';
        incomeToggleSlider.style.backgroundColor = '#ccc';
        incomeToggleCircle.style.transform = 'translateX(0)';
    }

    // Ensure form validates category input before submission
    const manualIncomeForm = document.getElementById('manual-income-form');
    if(manualIncomeForm){
        manualIncomeForm.addEventListener('submit', function(e) {
            const categoryInput = document.getElementById('category-input');
            if(!categoryInput.value || categoryInput.value.trim() === '') {
                e.preventDefault();
                alert('Please select or enter a category');
                categoryInput.focus();
                return false;
            }
        });
    }
</script>
<style>
    #income-input-mode-toggle:checked + span {
        background-color: #4CAF50 !important;
    }
    #income-input-mode-toggle:checked + span + span {
        transform: translateX(24px);
    }
</style>
</body>
</html>
<?php
$mysql->close();
?>
